/* tcleditline.c -- Command line editing for Tcl. */

/* David N. Welton <davidw@dedasys.com> */

/* $Id: counter.c,v 1.5 2005/01/20 18:58:12 davidw Exp $ */

#include <tcl.h>
#include <histedit.h>
#include <errno.h>
#include <string.h>

/* This carries around some information about the current state of the
 * library. */

typedef struct {
    EditLine *el;
    History *hi;
    HistEvent ev;
} ELChanInstance;

/*
 *-----------------------------------------------------------------------------
 *
 * TclELPrompt --
 *
 * 	This function is called to provide the prompt.  We could
 * 	probably do something fancy here.
 *
 * Results:
 *	A string to be used as the prompt.
 *
 * Side Effects:
 *	None.
 *
 *-----------------------------------------------------------------------------
 */

static char * TclELPrompt(EditLine *e) {
    Tcl_Interp *interp = NULL;
    el_get(e, EL_CLIENTDATA, interp);

    return "tcl> ";
}

static int
TclELcloseproc (ClientData instanceData, Tcl_Interp *interp) {
    ELChanInstance *instance = (ELChanInstance *)instanceData;
    /* Clean up our memory */
    history_end(instance->hi);
    el_end(instance->el);
    Tcl_Free((char *)instanceData);
    return 0;
}

static int
TclELinputproc(ClientData instanceData, char *buf,
	    int bufSize, int *errorCodePtr) {
    int count = 0;
    char *line = NULL;
    ELChanInstance *instance = (ELChanInstance *)instanceData;
    /* count is the number of characters read.  line is a const char*
       of our command line with the tailing \n */
    line = (char *)el_gets(instance->el, &count);

    if (count > bufSize) {
	el_push(instance->el, line+bufSize);
	count = bufSize;
    }

    strncpy(buf, line, count);

    if (count > 0) {
	/* Put the line into the history. */
	history(instance->hi, &(instance->ev), H_ENTER, line);
    }

    return count;
}

static int
TclELoutputproc(ClientData instancedata, CONST84 char *buf,
	     int toWrite, int *errorCodePtr)
{
    return EINVAL;
}

static int
TclELsetoptionproc(ClientData instanceData, Tcl_Interp *interp,
		 CONST char *optionName, CONST char *newValue) {
    ELChanInstance *instance = (ELChanInstance *)instanceData;

    if (strcmp(optionName, "-editor") == 0) {
	if (strcmp(newValue, "emacs") == 0) {
	    el_set(instance->el, EL_EDITOR, "emacs");
	} else if (strcmp(newValue, "vi") == 0) {
	    el_set(instance->el, EL_EDITOR, "vi");
	} else {
	    if (interp) {
		Tcl_AppendResult(interp, "bad value for -editor: ",
			"must be emacs, or vi",
			(char *) NULL);
		return TCL_ERROR;
	    }
	}
    } else if (strcmp(optionName, "-historysize") == 0) {
	int hsize = 0;
	if (Tcl_GetInt(interp, newValue, &hsize) != TCL_OK) {
	    return TCL_ERROR;
	}
	el_set(instance->el, H_SETSIZE, hsize);
    } else {
	return Tcl_BadChannelOption(interp, optionName, "editor historysize");
    }
    return TCL_OK;
}

static int
TclELgetoptionproc(ClientData instanceData, Tcl_Interp *interp,
		 CONST char *optionName, Tcl_DString *optionValue) {
    ELChanInstance *instance = (ELChanInstance *)instanceData;
    int retval = TCL_ERROR;


    if (optionName == NULL) {
	Tcl_DStringAppendElement(optionValue, "-editor");
    }
    if (optionName == NULL || strcmp (optionName, "-editor")) {
	char *result = NULL;
	el_get(instance->el, EL_EDITOR, result);
	Tcl_DStringAppendElement(optionValue, result);
	retval = TCL_OK;
    }
    if (optionName == NULL) {
	Tcl_DStringAppendElement(optionValue, "-historysize");
    }
    if (optionName == NULL || strcmp (optionName, "-historysize") == 0) {
	int hsize = 0;
	Tcl_Obj *sizeobj = NULL;

	hsize = history(instance->hi, &(instance->ev), H_GETSIZE);
	sizeobj = Tcl_NewIntObj(hsize);
	Tcl_DStringAppendElement(optionValue, Tcl_GetString(sizeobj));
	retval = TCL_OK;
    }

    return retval;
}

static void
TclELwatchproc(ClientData instancedata, int mask)
{
    return;
}

static int
TclELgethandleproc(ClientData instanceData, int direction,
		 ClientData *handlePtr) {
    /* FIXME - need a cleverer way of dealing with this. Since 'exec'
     * actually wants a fd, we give it the fd of the real stdin. */
    *handlePtr = 0;
    return TCL_OK;
}

static
Tcl_ChannelType ELChan = {
    "editline_channel",
    TCL_CHANNEL_VERSION_3,
    TclELcloseproc,
    TclELinputproc,
    TclELoutputproc,
    NULL,
    TclELsetoptionproc,
    TclELgetoptionproc,
    TclELwatchproc,
    TclELgethandleproc,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL
};

/*
 *-----------------------------------------------------------------------------
 *
 * Tcleditline_Init --
 *
 * 	Initialize the editline library stuff.  For now, this function
 * 	also swaps in its own version of stdin.
 *
 * Results:
 *	None.
 *
 * Side Effects:
 *	Replaces stdin with editline channel, deletes tcl_prompt1/2 variables.
 *
 *-----------------------------------------------------------------------------
 */

int
Tcleditline_Init(Tcl_Interp *interp) {
    Tcl_Channel chan;
    ELChanInstance *instancedata = NULL;
    EditLine *el = NULL;
    History *hi = NULL;
    HistEvent ev;

#ifdef USE_TCL_STUBS
    if (Tcl_InitStubs(interp, "8", 0) == NULL) {
	return TCL_ERROR;
    }
#endif

    Tcl_Eval(interp, "info nameofexecutable");
    el = el_init(Tcl_GetStringResult(interp), stdin, stdout, stderr);
    /* Associate the current interpreter with this editline, so that
     * the interpreter can be used in editline callbacks. */
    el_set(el, EL_CLIENTDATA, interp);
    /* Install editline signal handlers.  This is needed if we do things
     * like resize the window. */
    el_set(el, EL_SIGNAL, 1);
    el_set(el, EL_PROMPT, &TclELPrompt);
    el_set(el, EL_EDITOR, "emacs");

    hi = history_init();
    if (hi == 0) {
	Tcl_AppendResult(interp, "history could not be initialized", NULL);
	return TCL_ERROR;
    }

    /* Set the size of the history */
    history(hi, &ev, H_SETSIZE, 100);

    /* This sets up the callback function for history functionality. */
    el_set(el, EL_HIST, history, hi);

    instancedata = (ELChanInstance *)Tcl_Alloc(sizeof(ELChanInstance));

    instancedata->el = el;
    instancedata->hi = hi;
    instancedata->ev = ev;

    /* Make these empty so that we control the prompt ourselves. */
    Tcl_SetVar(interp, "::tcl_prompt1", "", 0);
    Tcl_SetVar(interp, "::tcl_prompt2", "", 0);

    /* Create the channel and register it as the standard input. */
    chan = Tcl_CreateChannel(&ELChan, "elchan", instancedata, TCL_READABLE);
    Tcl_SetStdChannel(chan, TCL_STDIN);
    Tcl_RegisterChannel(interp, chan);

    if ( Tcl_PkgProvide(interp, "editline", "0.1") != TCL_OK ) {
        return TCL_ERROR;
    }
    return TCL_OK;
}
